﻿using Hims.Shared.UserModels.Common;

namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.Library.Enums;
    using Hims.Shared.UserModels;
    using Hims.Shared.UserModels.DashboardWidget;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Shared.EntityModels;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/dashboard-widget")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DashboardWidgetController : BaseController
    {

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The admission services.
        /// </summary>
        private readonly IDashboardWidgetService dashboardWidgetService;

        /// <inheritdoc />
        public DashboardWidgetController(
            IAuditLogService auditLogServices,
            IDashboardWidgetService dashboardWidgetService)
        {
            
            this.auditLogServices = auditLogServices;
            this.dashboardWidgetService = dashboardWidgetService;
        }

        /// <summary>
        /// The fetch widgets.
        /// </summary>
        /// <param name="model">
        /// The widget filter model.
        /// </param>
        /// <returns>
        /// The list of Widgets.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of Widgets.
        /// - 500 - Problem with Server side code.
        /// </remarks>

        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAsync([FromBody] DashboardWidgetFilterModel model)
        {
            try
            {
                var widgets = await this.dashboardWidgetService.FetchAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = widgets
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The insert widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("insert-widget")]
        public async Task<ActionResult> InsertAsync([FromBody] DashboardWidgetModel model,[FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.dashboardWidgetService.InsertAsync(model).ConfigureAwait(false);

                if (response.Status == GenericStatus.Success)
                {
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.DashboardWidget,
                        LogFrom = (short)model.LogFrom,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $@"{model.CreatedByName} has added a new widget '{model.Name}' on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }

                return Ok(response);
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The update widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("update-widget")]
        public async Task<ActionResult> UpdateAsync([FromBody] DashboardWidgetModel model,[FromHeader]LocationHeader header)
        {
            try
            {
                var response = await this.dashboardWidgetService.UpdateAsync(model).ConfigureAwait(false);

                if (response.Status == GenericStatus.Success)
                {
                    var widgetTye = await this.dashboardWidgetService.FetchWidgetType(model.DashboardWidgetTypeId);
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.DashboardWidget,
                        LogFrom = (short)model.LogFrom,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $@"{model.ModifiedByName} has updated widget <br/>name: '{model.Name}'<br/>type: '{widgetTye}'<br/>stored procedure: '{model.StoredProcedureName}' on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }

                return Ok(response);
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The deactivate widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("deactivate-widget")]
        public async Task<ActionResult> DeactivateAsync([FromBody] DashboardActivateOrDeactivateWidgetModel model,[FromHeader]LocationHeader header)
        {
            try
            {
                var response = await this.dashboardWidgetService.DeactivateWidget(model.DashboardWidgetId, model.ModifiedBy, model.ModifiedByName);

                if (response > 0)
                {
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.DashboardWidget,
                        LogFrom = (short)model.LogFrom,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $@"{model.ModifiedByName} has deactivated widget <br/>'{model.Name}' on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }

                return Ok(new GenericResponse
                {
                    Status = response switch
                    {
                        -1 => GenericStatus.Error,
                        -2 => GenericStatus.Info,
                        _ => GenericStatus.Success
                    },
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The activate widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("activate-widget")]
        public async Task<ActionResult> ActivateAsync([FromBody] DashboardActivateOrDeactivateWidgetModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.dashboardWidgetService.ActivateWidget(model.DashboardWidgetId, model.ModifiedBy, model.ModifiedByName);

                if (response > 0)
                {
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.DashboardWidget,
                        LogFrom = (short)model.LogFrom,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $@"{model.ModifiedByName} has activated widget <br/>'{model.Name}' on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }

                return Ok(new GenericResponse
                {
                    Status = response switch
                    {
                        -1 => GenericStatus.Error,
                        -2 => GenericStatus.Info,
                        _ => GenericStatus.Success
                    },
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The execute stored procedure async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("execute-widget")]
        public async Task<ActionResult> ExecuteAsync([FromBody] ExecuteInputModel model)
        {
            try
            {
                var response = await this.dashboardWidgetService.ExecuteAsync(model);
                return Ok(response);
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }
    }
}